<?php
/* --------------------------------------------------------------
  StoredConfig.inc.php 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/

namespace StyleEdit\Entities;

/**
 * Class StoredConfig
 *
 * @package StyleEdit\Entities
 */
class StoredConfig extends Config
{
	/** @var string $filename */
	protected $filename;
	
	
	/**
	 * StoredConfig constructor.
	 *
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "name"-element with a non-empty
	 *                                   string value
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "boilerplate"-element with a
	 *                                   non-empty string value
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a valid "colorPalette"-element
	 *                                   containing an array of string values
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "creationDate"-element with a
	 *                                   formatted value "Y-m-d H:i:s"
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "modificationDate"-element with
	 *                                   a formatted value "Y-m-d H:i:s"
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "boilerplateModificationDate"-
	 *                                   element with a formatted value "Y-m-d H:i:s"
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "settings"-element
	 * @throws \InvalidArgumentException if $jsonData["settings"][]["icon"] is not a string
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a valid "customStyles"-element
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"] does not contain a valid
	 *                                   "entries"-element
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"]["entries"] does not contain a valid
	 *                                   "group"-element if $jsonData["settings"]["type"] is "display"
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"]["entries"] does not contain a valid
	 *                                   "position"-element if $jsonData["settings"]["type"] is "boxes"
	 * @throws \InvalidArgumentException if $jsonData["settings"]["entries"][]["value"] does not contain a
	 *                                   boolean value if $jsonData["settings"]["entries"][]["type"] is "switch"
	 * @throws \InvalidArgumentException if $jsonData["settings"]["entries"][]["value"] does not contain a
	 *                                   string value if $jsonData["settings"]["entries"][]["type"] is not "switch" or
	 *                                   "title"
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"]["entries"] does not contain an
	 *                                   "options" array if $jsonData["settings"]["entries"][]["type"] is "select"
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"]["entries"][]["options"] does not
	 *                                   contain string or array values
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"]["entries"] does not contain a valid
	 *                                   "tags"-element containing an array of strings
	 *
	 * @throws \InvalidArgumentException if $p_filename is not a string
	 *
	 * @param array  $jsonData
	 * @param string $p_filename
	 */
	public function __construct(array $jsonData, $p_filename)
	{
		parent::__construct($jsonData);
		
		$this->jsonData['isActive'] = $jsonData['isActive'];
		
		if(!is_string($p_filename))
		{
			throw new \InvalidArgumentException('$p_filename is not a string (' . gettype($p_filename) . ')');
		}
		
		$this->filename = $p_filename;
	}
	
	
	/**
	 * Returns the filename of the stored style configuration JSON-file
	 *
	 * @return string
	 */
	public function getFilename()
	{
		return $this->filename;
	}
	
	
	/**
	 * Returns active status of the style configuration (true or false)
	 *
	 * @return bool
	 */
	public function isActive()
	{
		return $this->jsonData['isActive'];
	}
	
	
	/**
	 * Set active status of the style configuration to true
	 *
	 * @return \StyleEdit\Entities\StoredConfig Same instance to make chained method calls possible.
	 */
	public function activate()
	{
		$this->jsonData['isActive'] = true;
		
		return $this;
	}
	
	
	/**
	 * Set active status of the style configuration to false
	 *
	 * @return \StyleEdit\Entities\StoredConfig Same instance to make chained method calls possible.
	 */
	public function deactivate()
	{
		$this->jsonData['isActive'] = false;
		
		return $this;
	}
	
	
	/**
	 * Sets the given JSON data array
	 *
	 * @param array $jsonData
	 *
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "name"-element with a non-empty
	 *                                   string value
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "boilerplate"-element with a
	 *                                   non-empty string value
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a valid "colorPalette"-element
	 *                                   containing an array of string values
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "creationDate"-element with a
	 *                                   formatted value "Y-m-d H:i:s"
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "modificationDate"-element with
	 *                                   a formatted value "Y-m-d H:i:s"
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "boilerplateModificationDate"-
	 *                                   element with a formatted value "Y-m-d H:i:s"
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a "settings"-element
	 * @throws \InvalidArgumentException if $jsonData["settings"][]["icon"] is not a string
	 * @throws \InvalidArgumentException if the given array $jsonData does not contain a valid "customStyles"-element
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"] does not contain a valid
	 *                                   "entries"-element
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"]["entries"] does not contain a valid
	 *                                   "group"-element if $jsonData["settings"]["type"] is "display"
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"]["entries"] does not contain a valid
	 *                                   "position"-element if $jsonData["settings"]["type"] is "boxes"
	 * @throws \InvalidArgumentException if $jsonData["settings"]["entries"][]["value"] does not contain a
	 *                                   boolean value if $jsonData["settings"]["entries"][]["type"] is "switch"
	 * @throws \InvalidArgumentException if $jsonData["settings"]["entries"][]["value"] does not contain a
	 *                                   string value if $jsonData["settings"]["entries"][]["type"] is not "switch" or
	 *                                   "title"
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"]["entries"] does not contain an
	 *                                   "options" array if $jsonData["settings"]["entries"][]["type"] is "select"
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"]["entries"][]["options"] does not
	 *                                   contain string or array values
	 * @throws \InvalidArgumentException if the given array $jsonData["settings"]["entries"] does not contain a valid
	 *                                   "tags"-element containing an array of strings
	 */
	public function setJsonDataArray(array $jsonData)
	{
		$this->_validateJsonDataStructure($jsonData);
		
		$this->jsonData = $jsonData;
	}
	
	
	/**
	 * Updates the filename. The same instance is returned to make chained method calls possible.
	 *
	 * @param string $p_filename
	 *
	 * @throws \InvalidArgumentException if $p_filename is not a string
	 *
	 * @return \StyleEdit\Entities\StoredConfig Same instance to make chained method calls possible.
	 */
	public function setFilename($p_filename)
	{
		if(!is_string($p_filename))
		{
			throw new \InvalidArgumentException('$p_filename is not a string (' . gettype($p_filename) . ')');
		}
		
		$this->filename = $p_filename;
		
		return $this;
	}
}